function [o_price, o_prices] = option_MLN_f(fvec, strike, rf, TTM, sigmavec,...
    weights, o_type)
%==========================================================================================
% This function computes the price of European options where the RND is a mixture
% of log-normal densities
%
% INPUT:
%        fvec          : a vector of forward/futures price of the underlying
%                        length(fvec) = number of log-normal densities in each mixture
%        strike        : strike price of the option. 
%        rf            : Annualized risk free rate, in decimals.
%        TTM           : Time to maturity, as a fraction of a year (1 year = 360 days).
%        sigmavec      : a vector of nnualized standard deviations of underlying asset's returns
%                      : corresponding to different log-normal densities in the mixture
%        weights       : a vector of weights for different log-normal densities in the mixture
%        o_type        : Option type, valid types are:
%                        'call'  - Call options
%                        'put'   - put options
%
% OUTPUT:
%        o_price       : Price of options
%        o_prices      : Unweighted Prices implied by different lognormal components
%                        (NxM matrix, where N is #obs, M is # mixture)
%==========================================================================================
% Option price is a weighted price of Black-Scholes option prices from each log-normal densities
%==========================================================================================
% This ver: 2021/10/16
% Authors: Yifan Li (yifan.li@manchester.ac.uk)
%          Ingmar Nolte (i.nolte@lancaster.ac.uk)
%          Manh Pham (m.c.pham@lancaster.ac.uk)
% Reference: Li, Y., Nolte, I., and Pham, M. C. (2021). Mixture-of-Lognormal 
%           Risk-Neutral Density Estimation Revisited: Asymptotics, Analytical Derivatives,
%           and the Mode Constraint
%========================================================================================== 


%===================
% Input checking
%===================

% fvec, sigmavec and weights
if ~isvector(fvec)
    error('fvec must be a vector.');
end

n = length(fvec);

if ~isvector(sigmavec) || ~isvector(weights) || ...
        length(sigmavec) ~= n || length(weights) ~= n
    error('sigmavec and weights must be vectors of same length as fvec');
end

% Replace NaN weights
NAweights = weights(isnan(weights));
if ~isempty(NAweights)
    weights(isnan(weights)) = (1 - sum(weights(~isnan(weights))))/length(NAweights);
end

if sum(weights) ~= 1 
    weights = weights/sum(weights);
end

% strike
if ~isvector(strike)
    error('strike must be a vector.');
end
nstrike = length(strike);

% o_type
if isempty(o_type)
    o_type = 'call';
end

if ~ismember(o_type, {'call', 'put'})
    error('o_type must be a string and one of: ''call'' or ''put''.');
end

% 

%===================
% Calculations
%===================

o_prices = nan(nstrike, n);


for i = 1:n
    price_i = ...
        BS1_f(fvec(i), 'forward', strike, rf, [], TTM, sigmavec(i) , o_type, false,false);
    o_prices(:,i) = price_i;
end

if size(weights,2) == 1
    weights = repmat(weights', nstrike, 1);
else
    weights = repmat(weights, nstrike, 1);
end

o_price = sum(o_prices.*weights, 2);


end
