function [y,yc,d,d2] = mixlognorm_f(x, muvec, sigmavec, weights)
%========================================================================================== 
% This function computes the weighted lognormal densities
%
% INPUT:
%        x        : A vector of values (i.e. expiry underlying price) at which the RND is evaluated.
%        mvec     : A vector of means of component lognormal densities
%        sigmavec : A vector of sigmas of component lognormal densities
%        weights  : A vector of weights of component lognormal densities
% Note that the individual mixtures are defined using the standard
% lognormal parametrization, i.e. 
%           f(x)=(x*sigma*sqrt(2pi))^-1*exp(-(log(x)-mu)^2/2/sigma^2) 
%
% OUTPUT: 
%        y        : Weighted lognormal density evaluated at xval
%        yc       : Weighted component density evaluated at xval
%        d        : Differentiated weighted lognormal density evaluated at xval
%        d2       : Twice-differentiated weighted component density evaluated at xval
%========================================================================================== 
% This ver: 2021/10/16
% Authors: Yifan Li (yifan.li@manchester.ac.uk)
%          Ingmar Nolte (i.nolte@lancaster.ac.uk)
%          Manh Pham (m.c.pham@lancaster.ac.uk)
% Reference: Li, Y., Nolte, I., and Pham, M. C. (2021). Mixture-of-Lognormal 
%           Risk-Neutral Density Estimation Revisited: Asymptotics, Analytical Derivatives,
%           and the Mode Constraint
%========================================================================================== 

%===================
% Input checking
%===================

% % x
if nargin < 1
    error('Input argument x is not defined.');
end

if ~isvector(x)
    error('x must be a vector.');
end

% % muvec
if nargin < 2 || isempty(muvec)
    muvec = 0;
end

if ~isvector(muvec)
    error('muvec must be a vector.');
end

% % sigmavec
if nargin < 3 || isempty(sigmavec)
    sigmavec = 1;
end

if ~isvector(sigmavec) || length(sigmavec) ~= length(muvec)
    error('sigmavec must be a vector of the same length as muvec.');
end

% % weights
if nargin < 4 || isempty(weights)
    weights = ones(size(muvec))*1/(length(muvec));
end

if ~isvector(weights) || length(weights) ~= length(muvec)
    error('weights must be a vector of the same length as muvec.');
end

% Replace NaN weights
NAweights = weights(isnan(weights));
if ~isempty(NAweights)
    weights(isnan(weights)) = (1 - sum(weights(~isnan(weights))))/length(NAweights);
end

% if sum(weights) ~= 1 
%     weights = weights/sum(weights);
% end

%===================
% Calculations
%===================
nmu = length(muvec);
nx  = length(x);
y   = nan(nx, nmu);
d   = nan(nx, nmu);
d2  = nan(nx, nmu);
if nx == 1
    y(1,:) = lognpdf(x, muvec, sigmavec);
    [d(1,:), d2(1,:)] = lognpdf_d(x, muvec, sigmavec);
else
    for i = 1:nmu
        y(:,i) = lognpdf(x, muvec(i), sigmavec(i));
        [d(:,i), d2(:,i)] = lognpdf_d(x, muvec(i), sigmavec(i));
    end
end

if size(weights,2) == 1
    weights = repmat(weights', nx, 1);
else
    weights = repmat(weights, nx, 1);
end

yc = y.*weights; % RND components
dc = d.*weights; 
dc2 = d2.*weights; % RND components
y = sum(yc,2); % RND
d = sum(dc,2); 
d2 = sum(dc2,2); 



