function spread = maxspread_f(oprices)
%========================================================================================== 
% Function to interpolate the maximum bid-ask spread for an option based on
% Bondarenko (2003, JoE) method (p. 110). 
%
% INPUT: 
%      oprices : a vector of option prices
% 
% OUTPUT:
%      spread  : a vector of corresponding interpolated maximum bid-ask spread
%========================================================================================== 
% This ver: 2021/10/16
% Authors: Yifan Li (yifan.li@manchester.ac.uk)
%          Ingmar Nolte (i.nolte@lancaster.ac.uk)
%          Manh Pham (m.c.pham@lancaster.ac.uk)
% Reference: Li, Y., Nolte, I., and Pham, M. C. (2021). Mixture-of-Lognormal 
%           Risk-Neutral Density Estimation Revisited: Asymptotics, Analytical Derivatives,
%           and the Mode Constraint
%========================================================================================== 


% Maximum bid-ask differentials (for group B) specified by CBOE - effective 7 Oct 2019
% (http://cdn.cboe.com/resources/regulation/circulars/regulatory/RG19-032.pdf)
% Note: These differentials only apply to quotes in open outcry and not for
% electronic quotes by market makers. However, we assume that these quotes
% are also relevant for all quotes
bidprice = [0; 1.99; 5; 10; 20; 50; 100; 200; 500]; 
maxspread = [0; 1; 1.6; 2; 4; 6; 10; 16; 24]/2;
% Here we assume that the max spread for a zero price is 0 
% and for a price >= 500 equals 24 (the
% maximum spread of the last bin)

% Bondarenko's max bid-ask spread
% bidprice = [0; 2; 5; 10; 20; 50]; 
% maxspread = [1/8; 1/4; 3/8; 1/2; 3/4; 1];


spread = interp1(bidprice, maxspread, oprices, 'linear');
if max(oprices) > bidprice(end)
    spread(oprices > bidprice(end)) = maxspread(end);
end

end