function [y,dy] = lognpdf_d(x,mu,sigma)
%==========================================================================================
%This function computes the derivatives of Lognormal probability density
%function (pdf).
%
%INPUT:
%   x: a column vector of values for the derivatives of the pdf to be
%      evaluated
%   mu and sigma: either scalars or vectors with the same dimension as
%      x, defined in the pdf:
%           f(x)=(x*sigma*sqrt(2pi))^-1*exp(-(log(x)-mu)^2/2/sigma^2) 
%      Note this is the standard parametrization of the lognormal pdf.
%
%OUTPUT:
%   y: first derivative of lognormal pdf evaluated at x
%   dy: second derivative lognormal pdf evaluated at x
%==========================================================================================
% This ver: 2021/10/16
% Authors: Yifan Li (yifan.li@manchester.ac.uk)
%          Ingmar Nolte (i.nolte@lancaster.ac.uk)
%          Manh Pham (m.c.pham@lancaster.ac.uk)
% Reference: Li, Y., Nolte, I., and Pham, M. C. (2021). Mixture-of-Lognormal 
%           Risk-Neutral Density Estimation Revisited: Asymptotics, Analytical Derivatives,
%           and the Mode Constraint
%========================================================================================== 
if nargin<1
    error(message('stats:lognpdf:TooFewInputs'));
end
if nargin < 2
    mu = 0;
end
if nargin < 3
    sigma = 1;
end

% Return NaN for out of range parameters.
sigma(sigma <= 0) = NaN;

% Negative data would create complex values, potentially creating spurious
% NaNi's in other elements of y.  Map them, and zeros, to the far right
% tail, whose pdf will be zero.
x(x <= 0) = Inf;

try
    y = (mu-sigma.^2-log(x)) .* exp(-0.5 * ((log(x) - mu)./sigma).^2) ./ (x.^2 .* sqrt(2*pi) .* sigma.^3);
    dy = (log(x).^2+(3*sigma.^2-2*mu).*log(x)+ 2 * sigma.^4 -(3*mu+1).*sigma .^2+mu.^2).* exp(-0.5 * ((log(x) - mu)./sigma).^2) ./ (x.^3 .* sqrt(2*pi) .* sigma.^5);
catch
    error(message('stats:lognpdf:InputSizeMismatch'));
end
