function [M_opt,info] = MLN_sequential_test(estpar_all,oprices,strike,rf, ...
    TTM,Ft,o_type,p,q,alpha,display)
%==========================================================================================
%This function performs the sequential test procedure of Li, Nolte and Manh
%(2021) to select the number of mixtures in the MLN model.
%
%INPUT:
%   estpar_all: a (M_max-1)-by-3*M_max matrix of parameters obtained from
%               the MLN_RND function.
%      oprices: N-by-1 option prices
%       strike: N-by-1 strike prices of the options
%           rf: risk-free rate
%          TTM: time to maturity of the options (in years)
%           Ft: underlying futures price
%       o_type: type of option, 'call' or 'put'.
%            p: polynomial degree for the auxiliary regression (default =
%               1)
%            q: degree of the Fourier series for the auxiliary regression
%               (default = 1)
%        alpha: significance level of the Wald test (default = 0.01)   
%      display: true or false. Toggle display of test results to the
%               command window (default = true)
%
%OUTPUT:
%    M_opt: chosen M based on the sequential test procedure
%     info: diagnostic information for the tests performed
%==========================================================================================
% This ver: 2021/10/16
% Authors: Yifan Li (yifan.li@manchester.ac.uk)
%          Ingmar Nolte (i.nolte@lancaster.ac.uk)
%          Manh Pham (m.c.pham@lancaster.ac.uk)
% Reference: Li, Y., Nolte, I., and Pham, M. C. (2021). Mixture-of-Lognormal 
%           Risk-Neutral Density Estimation Revisited: Asymptotics, Analytical Derivatives,
%           and the Mode Constraint
%========================================================================================== 

%default vlaues for the HAC-robust Wald test
if nargin<8
    p=1;
    q=1;
    alpha=0.01;
    display=true;
elseif nargin<10
    display=true;
end


Nobs=length(oprices);
[M_max,~]=size(estpar_all);
info=zeros(M_max-1,3);
M_opt=2;
for i=1:M_max
    M=i+1;
    estpar=estpar_all(i,1:3*M)';
    o_fit = option_MLN_f(estpar(M+1:2*M), strike, rf, TTM, estpar(2*M+1:3*M), estpar(1:M), o_type);
    resid = oprices - o_fit;   
    a = [ones(M,1); zeros(2*M,1)];
    B = [zeros(M,3*M); eye(M) zeros(M,2*M); zeros(M,3*M)];
    Jh = [a, (B+B')*estpar]';   
    JO=MLN_OptionGrad_f(estpar,strike,rf,TTM,o_type);%N-by-3M Jacobian matrix
    G_prel=JO'*JO/Nobs;%pre-limiting version of G matrix
    Ginv_prel=inv(G_prel);
    M_prel = Ginv_prel - Ginv_prel*Jh'*inv(Jh*Ginv_prel*Jh')*Jh*Ginv_prel;
    % Get the OLS estimator for beta
    C = MLN_FFF(log(strike./Ft),p,q);
    bhat=C\resid;
    a0=inv(C'*C)*(C'-C'*JO*M_prel*JO');
    Vwald=covnw_f(a0'.*resid);
    % Compute Wald test stat and p-value
    Wald_stat = Nobs*bhat'*inv(Vwald+eye(length(bhat))*1e-5)*bhat;
    Wald_pval = chi2cdf(Wald_stat,p+1+q*2,'upper');
    M_opt=M_opt+(Wald_pval<=alpha);
    info(i,:)=[M Wald_stat Wald_pval];
end
info=array2table(info,'VariableNames',{'Mhat','Wald_stat','p_val'});
M_opt=min(M_max+1,M_opt);
if display
    disp(info)
end
end

