function par_sorted = MLN_parm_sort(estpar,sortby,Ft)
%==========================================================================================
%This function sorts the 3M-by-1 MLN parameter estimates based on w,
%F or sigma.
%
%INPUT:
%   estpar: 3M-by-k parameter vector of the format
%  [w^(1), ... , w^(M), F^(1), ... , F^(M), sigma^(1), ... , sigma^(M)]
%   sortby: integer values in [-3 -2 -1 1 2 3]. Negative (positive) value indicates
%   descending (ascending) sort and 1, 2, 3 represents sorting based on w,
%   F or sigma.
%
%OUTPUT:
%   par_sorted: 3M-by-1 sorted parameters
%==========================================================================================
% This ver: 2021/10/16
% Authors: Yifan Li (yifan.li@manchester.ac.uk)
%          Ingmar Nolte (i.nolte@lancaster.ac.uk)
%          Manh Pham (m.c.pham@lancaster.ac.uk)
% Reference: Li, Y., Nolte, I., and Pham, M. C. (2021). Mixture-of-Lognormal 
%           Risk-Neutral Density Estimation Revisited: Asymptotics, Analytical Derivatives,
%           and the Mode Constraint
%========================================================================================== 

[T,K]=size(estpar);

M=ceil(T/3);

if ~((mod(length(estpar),3)==0) || (mod(length(estpar),3)==1))
    error('estpar must be a 3M-by-k or a (3M-2)-by-k vector.')
end
if mod(T,3)==1
    estpar=MLN_parm_convert(estpar,Ft);
end

if ismember(sortby, [1 2 3])
    direct = 'ascend';
elseif ismember(sortby, [-1 -2 -3])
    direct = 'descend';
else
    error('sortby must be an element in [-3 -2 -1 1 2 3]')
end
par_sorted=zeros(T,K);
for i=1:K
    parmat=sortrows(reshape(estpar(:,i),M,3),abs(sortby),direct);
    temp = reshape(parmat,3*M,1);
    if mod(T,3)==1
        par_sorted(:,i)=MLN_parm_convert(temp);
    elseif mod(T,3)==0
        par_sorted(:,i)=temp;
    end
end

end

