function [C, Ceq, GC, GCeq] = MLN_nlcon(x, Ft,TTM,n_mode,n_mode_step)
%==========================================================================================
% The function constructs nonlinear constraints for the local optimization
% of the MLN parameters, including the mode constraint
%
%INPUT:
%        x             : (3M-2)-by-1 vector of parameter estimates of the format:
%       [sigma^(1), ... , sigma^(M), w^(1), ... , w^(M-1), F^(1), F^(2), ... , F^(M-1)]
%        Ft            : spot futures price
%        TTM           : time-to-maturty in years
%        n_mode        : maximum number of modes in the mode constraint
%        n_mode_step   : number of steps S used to compute the
%                        number of inflection points
%
%OUTPUT:
%        C             : value of the nonlinear inequality constraint
%        Ceq           : value of the nonlinear equality constraint
%        GC            : analytical gradient of the nonlinear inequality
%                        constraint
%        GCeq          : analytical gradient of the nonlinear equality
%                        constraint
%==========================================================================================
% This ver: 2021/10/16
% Authors: Yifan Li (yifan.li@manchester.ac.uk)
%          Ingmar Nolte (i.nolte@lancaster.ac.uk)
%          Manh Pham (m.c.pham@lancaster.ac.uk)
% Reference: Li, Y., Nolte, I., and Pham, M. C. (2021). Mixture-of-Lognormal 
%           Risk-Neutral Density Estimation Revisited: Asymptotics, Analytical Derivatives,
%           and the Mode Constraint
%========================================================================================== 


n = (length(x) + 2)/3; % number of mixtures
w_vec = [x(1+n:2*n-1); 1 - sum(x(1+n:2*n-1))];
f_vec = [x(2*n:end); (Ft - x(2*n:end)'*w_vec(1:end-1))/w_vec(end)];
sigma_vec = x(1:n);


if size(x,1) == 1
    x = x'; % x is used as a column vector
end

% C and GC
% Constraints on F: % w_1*F_1 + ... + w_{n-1}*F_{n-1} <= F
C(1) = x(n+1:2*n-1)'*x(2*n:3*n-2) - Ft;  % w_1*F_1 + ... + w_{n-1}*F_{n-1} <= F
GC(:,1) = [zeros(n,1); x(2*n:3*n-2); x(n+1:2*n-1)];

% mode constraint
% compute the possible range of the modes for an MLN density
sigma_vec=sigma_vec.*sqrt(TTM);
mu_vec=log(f_vec)-0.5*sigma_vec.^2;
A_l=exp(-sigma_vec.*sqrt(sigma_vec.^2+4)/2-3*sigma_vec.^2/2+mu_vec);
A_h=exp(sigma_vec.*sqrt(sigma_vec.^2+4)/2-3*sigma_vec.^2/2+mu_vec);
[~,~,~,dRND2]=MLN_RND_PDF(x,linspace(min(A_l),max(A_h),n_mode_step),TTM,Ft);
C(2) = sum(abs(diff(dRND2>=0)))-(2*n_mode); % mode constraint
GC(:,2) = zeros(3*n-2,1);

% Ceq and GCeq are empty as no nonlinear equality constraint is imposed
Ceq = []; GCeq = [];

end