function [residinfo,parminfo,RNDinfo] = MLN_RND_infer(estpar, oprices, strike,rf,TTM,Ft,o_type,xval,Disp)
%==========================================================================================
%This function computes the post-estimation statistics and RND estimate
%based on an estimated MLN parameter vector
%
%INPUT:
%       estpar: 3M-by-1 parameter vector of an MLN model
%      oprices: observed option prices
%       strike: strikes of the option
%           rf: risk-free rate of hte options
%          TTM: time-to-maturity of options
%           Ft: underlying futures price
%       o_type: type of options, 'call' or 'put'
%         xval: vector of points to evaluate the RND estimates
%         Disp: toggle display setting of outputs. Default = false
%
%OUTPUT:
%   residinfo: residual information of the constrained NLS regression.
%             o_fit -- fitted option prices
%             resid -- fitted NLS residuals
%             rmse  -- root mean squared errror of residuals
%    parminfo: parameter information
%           estpar  -- estimated parameter vector
%          SigmaNW  -- estimated Newey-West variance covariance matrix
%           parm_V  -- variance-covariance matrix of parameter estimates
%           parm_se -- standard errors of parameter estimates
%     RNDinfo: RND estimate info
%               PDF -- density function of the RND evaluated at xval
%              dRND -- first-order derivative of the density function
%             d2RND -- second-order derivative of the density function
%          modeinfo -- number of critical points, inflection points, and
%          the interval [A^-, A^+] on which the derivatives are evaluated.
%==========================================================================================
% This ver: 2021/10/16
% Authors: Yifan Li (yifan.li@manchester.ac.uk)
%          Ingmar Nolte (i.nolte@lancaster.ac.uk)
%          Manh Pham (m.c.pham@lancaster.ac.uk)
% Reference: Li, Y., Nolte, I., and Pham, M. C. (2021). Mixture-of-Lognormal 
%           Risk-Neutral Density Estimation Revisited: Asymptotics, Analytical Derivatives,
%           and the Mode Constraint
%========================================================================================== 

if nargin<9
    Disp= 0;
end

estpar=packr(estpar);%remove nan from the parameter vector
M=length(estpar)/3;
N=length(oprices);

residinfo=struct;
RNDinfo=struct;
parminfo=struct;

%sort the mixture parameters according to F, ascending
estpar = MLN_parm_sort(estpar,2);
%computing residual statistics
residinfo.o_fit = option_MLN_f(estpar(M+1:2*M ), strike, rf, TTM, estpar(2*M+1:3*M), estpar(1:M),o_type);
residinfo.resid = oprices - residinfo.o_fit;
residinfo.sse  = sum(residinfo.resid.^2);
%extract RND information
[RNDinfo.PDF,RNDi,~,~]   = MLN_RND_PDF(estpar,xval,TTM, Ft);
sigma_vec=estpar(2*M+1:3*M).*sqrt(TTM);
f_vec=estpar(M+1:2*M);
mu_vec=log(f_vec)-0.5*sigma_vec.^2;
A_l=min(exp(-sigma_vec.*sqrt(sigma_vec.^2+4)/2-3*sigma_vec.^2/2+mu_vec));
A_h=max(exp(sigma_vec.*sqrt(sigma_vec.^2+4)/2-3*sigma_vec.^2/2+mu_vec));
[~,~,dRND,d2RND]   = MLN_RND_PDF(estpar,linspace(A_l,A_h,1e4)',TTM, Ft);
n_criticalpoints = sum(abs(diff(dRND>=0)));
n_inflectionpoints = sum(abs(diff(d2RND>=0)));
RNDinfo.modeinfo= [n_criticalpoints n_inflectionpoints A_l A_h];

%Construct confidence bounds for parameters and RND
JO=MLN_OptionGrad_f(estpar,strike,rf,TTM,o_type);%N-by-3M Jacobian matrix
G_prel=JO'*JO/N;%pre-limiting version of G matrix
Ginv_prel=inv(G_prel);
a = [ones(M,1); zeros(2*M,1)];
B = [zeros(M,3*M); eye(M) zeros(M,2*M); zeros(M,3*M)];
Jh = [a, (B+B')*estpar]';
%pre-limiting version of M matrix
M_prel = Ginv_prel - Ginv_prel*Jh'*inv(Jh*Ginv_prel*Jh')*Jh*Ginv_prel;
M_prel = (M_prel+M_prel')/2;
Sigma_NW = covnw_f(JO.*residinfo.resid,[],0); %Newey-West estimation of the sigma matrix
parminfo.SigmaNW=Sigma_NW;
parminfo.estpar=estpar;
parminfo.parm_V = M_prel*Sigma_NW*M_prel/N;
parminfo.parm_V = (parminfo.parm_V+parminfo.parm_V')/2;
parminfo.parm_se= sqrt(diag(parminfo.parm_V));
parminfo.t_stat = estpar./parminfo.parm_se;
parminfo.p_val  = 2*normcdf(abs(parminfo.t_stat),'upper');
parminfo.parm_out = table(estpar,parminfo.parm_se,parminfo.t_stat,parminfo.p_val,  ...
    'VariableNames',{'ParmEst','StdErr','t_stat','p_val'}, ...
    'RowNames',['w'+string(1:M) 'F'+string(1:M) 'sigma'+string(1:M)]');

%compute the numerical Jacobian of the RND estimates w.r.t. estpar
f=@(x) MLN_RND_PDF(x,xval,TTM);
Jp=gradient_f(f,estpar,1e-6);
RND_V=Jp*parminfo.parm_V*Jp';
RND_se=sqrt(diag(RND_V));
%construct 95% confidence bounds
RNDinfo.RND_bounds=[RNDinfo.PDF-1.97*RND_se  RNDinfo.PDF+1.97*RND_se];
RNDinfo.dRND=dRND;
RNDinfo.d2RND=d2RND;

if nargout < 1 || Disp == 1
    subplot(1,2,1)
    h=plot(xval,[RNDinfo.PDF RNDinfo.RND_bounds]);
    h(1).Color='k';
    h(2).Color='r';h(2).LineStyle='--';
    h(3).Color='r';h(3).LineStyle='--';
    hold on
    plot(xval, RNDi, 'b--', 'LineWidth',0.5);
    legend({'Estimated RND','95% Confidence Bounds'})
    ylim([0 max(RNDinfo.PDF*1.5)])
    
    subplot(1,2,2)
    plot(strike,residinfo.resid,'o');
    title('Residuals')
    
    disp(parminfo.parm_out)
end
end

function y=packr(x)
% PACKR
% PACKR(x) deletes all the rows of x that contain a MATLAB missing element (NaN)
t=isnan(x);
c=sum(t,2);
y=x(c==0,:);
end