function [estpar,fval_best,flag_best,Time] = MLN_RND_core(oprice,strike,rf,TTM,F_t,M,o_type,...
    x0,n_mode,fval_ub,options)
%==========================================================================================
%The core function that calculates the MLN estimate of the RND based on a
%cross-section of option prices with a particular choice of number of
%mixtures M using the global search algorithm documented in Appendix B of
%Li, Nolte and Pham (2021).
%
%INPUT:
%       oprice: N-by-1 option prices
%       strike: N-by-1 strike prices of the options
%           rf: risk free rate
%          TTM: time to maturity of the options in years
%          F_t: underlying futures price
%            M: number of mixtures for the MLN method
%       o_type: type of option, 'call' or 'put'.
%           x0: starting value for the parameter estimate
%       n_mode: number of maximum modes for the MLN density
%      fval_ub: upper bound of the objective function
%      Options: options to fine-tune the MLN-based RND estimation. If not
%            provided, default options are used. See the function MLN_options for
%            details.
%
%OUTPUT:
%   estpar: 3M-by-1 estimated parameter vector
%   fval_best: objective function at the minimized parameter vector  
%   flag_best: exit flag of the minimized parameter vector. 
%           = 1 or 2: possible local minima
%           = -1    : maximum iteration time reached but still a valid solution
%           = -Inf  : maximum allowed time used up
%           = nan   : (mode constraint only) no estimation performed
%   Time: estimation time in seconds
%==========================================================================================
% This ver: 2021/10/16
% Authors: Yifan Li (yifan.li@manchester.ac.uk)
%          Ingmar Nolte (i.nolte@lancaster.ac.uk)
%          Manh Pham (m.c.pham@lancaster.ac.uk)
% Reference: Li, Y., Nolte, I., and Pham, M. C. (2021). Mixture-of-Lognormal 
%           Risk-Neutral Density Estimation Revisited: Asymptotics, Analytical Derivatives,
%           and the Mode Constraint
%========================================================================================== 


% Unpack the options
analyticGrad = options.analyticGrad;
analyticHess = options.analyticHess;
n_mode_step = options.n_mode_step;
max_time = options.max_time_iter;
max_time_M = options.max_time_M;
Nbest_max= options.Nbest_max;

% Tuning parameter for the initial value randomizer
Lambda=0.9;
% Starting value for sigma as the ATM BSIV
i_ATM=abs(strike-F_t)==min(abs(strike-F_t));
sig_start=blkimpv(F_t,strike(i_ATM),rf,TTM,oprice(i_ATM),[],[],o_type);


% Set up restrictions and options for fmincon
TolX=1e-8; TolFun=1e-10;
% Lower bound, upper bounds and constraints
lb=ones(3*M-2,1)*1e-3; % Lower bound
ub=[10*ones(M,1); ones(M-1,1); 1e6*ones(M-1,1)]; % Upper bound
A=[zeros(M,1); ones(M-1,1); zeros(M-1,1)]'; b = 1; % linear inequality: w_1 + ... + w_{M-1} < 1
Aeq = []; beq = []; % linear equality
nlcon = @(x) MLN_nlcon(x, F_t,TTM,n_mode,n_mode_step); % nonlinear inequality w_1*F_1 + ... + w_{M-1}*F_{M-1} <= F
% & mode constraint

% Objective function
f = @(x) MLN_MSE(x, M, oprice, strike, rf, TTM, F_t, o_type, analyticGrad, analyticHess);


%==================
% Matrix to store outputs
out = []; %saving all feasible parameter estimates
Nobs = length(oprice);
tstart_all=tic;
Ncount=0;
isvalid=[false false];
while ~isvalid(1)
    parstart_0 = MLN_x0_gen(1,M,strike,F_t,sig_start,TTM,n_mode,n_mode_step);
    if  length(x0)/3==M-1
        estpar_q=MLN_parm_extend(x0,F_t,sig_start);
    else
        error('Dimension of x0 incorrect')
    end
    parstart=estpar_q*Lambda+parstart_0*(1-Lambda);
    isvalid=nlcon(parstart)<=0;
end
estpar_best=parstart;
fval_best=Inf;
OT_flag=0;
while Ncount<Nbest_max
    fmincon_options = optimoptions(@fmincon,'Algorithm','interior-point',...
        'SpecifyObjectiveGradient',analyticGrad,'SpecifyConstraintGradient',analyticGrad,...
        'ScaleProblem',true,...
        'OptimalityTolerance',TolFun,'StepTolerance',TolX,...
        'MaxFunctionEvaluation',1e8,'MaxIter',1e8,'Display','off');
    if analyticHess
        fmincon_options.HessianFcn=@(x,lambda) MLN_hessFcn(x,lambda,M,oprice,strike,rf,TTM,F_t,o_type);
    end
    tstart=tic;
    % option to restrict estimation time to max_time (in seconds)
    fmincon_options.OutputFcn = @(x,optim,state) CLS_timer(x,optim,state,tstart,max_time);
    [estpar,fval,flag] = fmincon(f,parstart,A,b,Aeq,beq,lb,ub,nlcon,fmincon_options);
    fval=fval*2*Nobs;%standardize the objective function to sum of squared errors
    % update the best parameter estimates and objective function value
    if fval<=fval_best && flag~=-2
        estpar_best=estpar;
        fval_best=fval;
    end
    % keep the feasible local minima
    if  fval <= fval_ub
        if flag > 0 || (flag == -1 && (A*estpar-b <= 0 && all(nlcon(estpar) <= 0)))
            Ncount=Ncount+1;
            out=[out [flag;toc(tstart);estpar; fval]];
            if strcmp(options.Display,'local')
                fprintf('Ncount = %d, fval = %.2e, fval_best = %.2e, fval_ub = %.2e, total time %.2f seconds \n',Ncount,fval, fval_best,fval_ub,toc(tstart_all))
            end
        end
    end
    
    
    % perturb the best parameter estimates as the initial point
    isvalid=[false false];
    while ~isvalid(1)
        parstart=estpar_best*Lambda+MLN_x0_gen(1,M,strike,F_t,sig_start,TTM,n_mode,n_mode_step)*(1-Lambda);
        isvalid=nlcon(parstart)<=0;
    end
    % break if the maximum allowed time is used-up
    Time=toc(tstart_all);
    if Time > max_time_M
        OT_flag=1;
        break
    end
end
if ~isempty(out)
    [fval_best, j_best] = min(out(end,:));
    estpar=out(3:end-1,j_best); %pick the best parameter estimate
    estpar=MLN_parm_convert(estpar,F_t);%convert the parameter estimates to the 3M-by-1 format
    if OT_flag == 0
        flag_best=out(1,j_best);
    elseif OT_flag == 1
        flag_best= -Inf;%set flag_best to -Inf if OT_flag == 1
    end
else
    warning('No valid parameter estimates returned in the allowed time limit.')
    estpar = nan(3*M,1);
    flag_best = -Inf;
    fval_best = nan;
end

end
