function [RND,RNDc,dRND,d2RND] = MLN_RND_PDF(parvec, xval,TTM,Ft)
%==========================================================================================
%The function computes the PDF and derivatives of the MLN_RND for a particular
%choice of parameter vector evaluated at xval with TTM denoting the time to maturity.
%
%INPUT:
%   parvec: either 3M-by-1 or (3M-2)-by-1. When it is (3M-2)-by-1:
%         [sigma^(1), ... , sigma^(M), w^(1), ... , w^(M-1), F^(1), ... , F^(M-1)]
%       When it is 3M-by-1:
%         [w^(1), ... , w^(M), F^(1), ... , F^(M), sigma^(1), ... , sigma^(M)]
%   xval: points to evaluate the pdf and the derivatives of the MLN
%         density, must be a column vector
%   TTM: time to maturity of the options (in years)
%   Ft: underlying futures price
%
%OUTPUT:
%      RND: the PDF of MLN_RND evaluated at xval
%     RNDc: the mixture-wise PDF of MLN_RND
%     dRND: the first-order derivative of RND
%    d2RND: the second-order derivative of RND
%==========================================================================================
% This ver: 2021/10/16
% Authors: Yifan Li (yifan.li@manchester.ac.uk)
%          Ingmar Nolte (i.nolte@lancaster.ac.uk)
%          Manh Pham (m.c.pham@lancaster.ac.uk)
% Reference: Li, Y., Nolte, I., and Pham, M. C. (2021). Mixture-of-Lognormal 
%           Risk-Neutral Density Estimation Revisited: Asymptotics, Analytical Derivatives,
%           and the Mode Constraint
%========================================================================================== 

if nargin<3
    Ft=nan;
end

switch mod(length(parvec),3)
    case 1  
        M=(length(parvec)+2)/3;
        sigmavec = parvec(1:M);
        if M > 1
            weights = [parvec(1+M:2*M-1); 1 - sum(parvec(1+M:2*M-1))];
            fvec = [parvec(2*M:end); (Ft - parvec(2*M:end)'*weights(1:end-1))/weights(end)];
        else
            weights = 1; fvec = Ft;
        end
    case 0
        M=length(parvec)/3;
        weights = parvec(1:M);
        fvec = parvec(M+1:2*M);
        sigmavec = parvec(2*M+1:3*M);
    otherwise
       error('Incorrect dimension of the parameter vector.')  
end
[RND, RNDc,dRND,d2RND] = mixlognorm_f(xval, log(fvec)-0.5*sigmavec.^2.*TTM,...
    sigmavec.*sqrt(TTM), weights); %convert to the standard lognormal parametrization here
end

