function [estpar_all,estpar_all_mc,estpar_info] = MLN_RND(oprice,strike,rf,TTM,F_t,o_type,M_max,...
    n_mode,options)
%==========================================================================================
%The function estimate the parameters of the MLN-based RND based on a
%cross-section of option prices using the global search algorithm + mode
%constraint based on Li, Nolte and Pham (2021).
%
%INPUT:
%   oprice: N-by-1 option prices
%   strike: N-by-1 strike prices of the options
%   rf: risk-free rate
%   TTM: time to maturity of the options (in years)
%   F_t: underlying futures price
%   o_type: type of option, 'call' or 'put'.
%   M_max: maximum number of mixtures for the MLN method
%   n_mode: number of maximum modes for the MLN density
%   Options: options to fine-tune the MLN-based RND estimation. If not
%   provided, default options are used. See the function MLN_options for
%   details.
%
%OUTPUT:
%   estpar_all: (M_max-1)-by-(M_max*3) matrix of estimated parameter 
%       vector without mode constraint. Each row reports the estimated
%       parameter from M=2 to M=M_max, with redundant elements of each row
%       being set to NaN.
%   estpar_all_mc: (M_max-1)-by-(M_max*3) matrix estimated parameter 
%       vector with mode constraint. Each row reports the estimated
%       parameter from M=2 to M=M_max, with redundant elements of each row
%       being set to NaN.
%   estpar_info: diagnostic information of the converged algorithm. It
%       takes the format of an (M_max-1)-by-10 matrix. Each row reports 
%       the diagnostic information from M=2 to M=M_max. The entries for 
%       the first 5 element of each row are:
%          (1) fval: objective function at the minimized parameter vector
%          (2) flag: exit flag of the minimized parameter vector.
%                   = 1 or 2: possible local minima
%                   = -1    : maximum iteration time reached but still a valid solution
%                   = -Inf  : maximum allowed time used up
%                   = nan   : (mode constraint only) no estimation performed
%          (3) Time: time spent to reach the solution
%          (4) n_criticalpoints: number of critical points in the estimated RND
%          (5) n_inflectionpoints: number of inflection points in the estimated RND
%          The next 5 elements are the same as the first 5 except with an active mode
%          constraint.
%==========================================================================================
% This ver: 2021/10/16
% Authors: Yifan Li (yifan.li@manchester.ac.uk)
%          Ingmar Nolte (i.nolte@lancaster.ac.uk)
%          Manh Pham (m.c.pham@lancaster.ac.uk)
% Reference: Li, Y., Nolte, I., and Pham, M. C. (2021). Mixture-of-Lognormal 
%           Risk-Neutral Density Estimation Revisited: Asymptotics, Analytical Derivatives,
%           and the Mode Constraint
%========================================================================================== 

%use the default options if no options are provided
if nargin<9
    options=MLN_options();
end
%Initializing outputs
estpar_all=zeros(M_max-1,M_max*3);
estpar_all_mc=zeros(M_max-1,M_max*3);
estpar_info=zeros(M_max-1,10);

%ATM BS implied volatility as initial values
i_ATM=abs(strike-F_t)==min(abs(strike-F_t));
sig_start=blkimpv(F_t,strike(i_ATM),rf,TTM,oprice(i_ATM),[],[],o_type);


fval_ub=Inf(M_max-1,2);
estpar=nan;estpar_mc=nan;
Mhat=2;
while Mhat <= M_max
    if Mhat==2
        x0=[1;F_t;sig_start];
        x0_mc=x0;
    else
        x0=estpar;
        x0_mc=estpar_mc;
    end
    [estpar,fval,flag,Time] = MLN_RND_core(oprice,strike,rf,TTM,F_t,Mhat,o_type,x0,Mhat,fval_ub(Mhat-1,1),options); 
    fval_ub(Mhat,1)=fval;
    estpar_all(Mhat-1,:)=[estpar'  nan(1,(M_max-Mhat)*3)];
    
    %check if a mode constraint is required
    sigma_vec=estpar(2*Mhat+1:3*Mhat).*sqrt(TTM);
    f_vec=estpar(Mhat+1:2*Mhat);
    mu_vec=log(f_vec)-0.5*sigma_vec.^2;
    A_l=min(exp(-sigma_vec.*sqrt(sigma_vec.^2+4)/2-3*sigma_vec.^2/2+mu_vec));
    A_h=max(exp(sigma_vec.*sqrt(sigma_vec.^2+4)/2-3*sigma_vec.^2/2+mu_vec));
    [~,~,dRND,d2RND]   = MLN_RND_PDF(estpar,linspace(A_l,A_h,1e4)',TTM, F_t);
    n_criticalpoints = sum(abs(diff(dRND>=0)));
    n_inflectionpoints = sum(abs(diff(d2RND>=0)));
    if strcmp(options.Display,'global') || strcmp(options.Display,'local')
        fprintf('M = %d without mode constraint completed in %4.2f seconds \n',Mhat,Time)
    end
    if n_inflectionpoints>2*n_mode
        [estpar_mc,fval_mc,flag_mc,Time_mc] = MLN_RND_core(oprice,strike,rf,TTM,F_t,Mhat,o_type,x0_mc,n_mode,fval_ub(Mhat-1,2)*1.1,options);
        fval_ub(Mhat,2)=fval_mc;
        if strcmp(options.Display,'global') || strcmp(options.Display,'local')
            fprintf('M = %d with at most %d modes completed in %4.2f seconds \n',Mhat,n_mode,Time_mc)
        end
    else
        if strcmp(options.Display,'global') || strcmp(options.Display,'local')
            fprintf('M = %d with at most %d modes skipped \n',Mhat,n_mode)
        end
        estpar_mc=estpar;
        fval_mc=fval;
        Time_mc=nan;
        fval_ub(Mhat,2)=fval_mc;
        flag_mc=nan; 
    end
    
    %compute the critical and inflection points of the RND with mode
    %constraint
    sigma_vec=estpar_mc(2*Mhat+1:3*Mhat).*sqrt(TTM);
    f_vec=estpar_mc(Mhat+1:2*Mhat);
    mu_vec=log(f_vec)-0.5*sigma_vec.^2;
    A_l=min(exp(-sigma_vec.*sqrt(sigma_vec.^2+4)/2-3*sigma_vec.^2/2+mu_vec));
    A_h=max(exp(sigma_vec.*sqrt(sigma_vec.^2+4)/2-3*sigma_vec.^2/2+mu_vec));
    [~,~,dRND_mc,d2RND_mc]   = MLN_RND_PDF(estpar_mc,linspace(A_l,A_h,1e4)',TTM, F_t);
    n_criticalpoints_mc = sum(abs(diff(dRND_mc>=0)));
    n_inflectionpoints_mc = sum(abs(diff(d2RND_mc>=0)));
    
    
    estpar_all_mc(Mhat-1,:)=[estpar_mc'  nan(1,(M_max-Mhat)*3)];
    estpar_info(Mhat-1,:)=[fval flag Time n_criticalpoints n_inflectionpoints fval_mc flag_mc Time_mc n_criticalpoints_mc n_inflectionpoints_mc];
    Mhat=Mhat+1;
end
end

