function  [mse,grad,hess,gradObs,fprices,fprice, resid]= MLN_MSE(parm,nmixture,oprice,K, rf, TTM, ...
    ft, o_type, analyticGrad, analyticHess)
%==========================================================================================
% This function computes the mean squared difference between market futures option prices 
% and their theoretical prices based on a mixture of log-normal densities
%
% INPUT:
%        parm        : (3M-2)-by-1 vector of parameters (sigma, weights, Futures) of the log-normal densities 
%                        in a mixture
%                        parm =[sigma_1, sigma_2, ..., sigma_M,  
%                                       w_1, w_2, ..., w_{M-1},
%                                       F_1, F_2, ..., F_{M-1}], 
%        nmixture      : number of component densities in a mixture. Default = 2
%        oprice        : vector of option prices
%        K             : vector of strike prices
%        rf            : scalar of annualized risk free rate, in decimals.
%        TTM           : scalar of time to maturity in years.
%        ft            : spot futures price
%        o_type        : Option type, valid types are:
%                        'call'  - Call options [default]
%                        'put'   - put options
%      analyticGrad    : Logical, whether analytical Gradients are calculated
%                        If analyticGrad = false, grad = gradObs = []
%      analyticHess    : Logical, whether analytical Hessian is calculated
%                        If analyticHess = false, hess = []
%
% OUTPUT: 
%        mse           : mean squared difference
%        grad          : Analytic gradient vector (d MSE/ d theta)
%        hess          : Analytic Hessian metrix (d^2 MSE/ d theta^2)
%        gradObs       : Analytic per-obs gradient matrix 
%        fprices       : fitted option prices per mixture
%        fprice        : fitted option prices
%        resid         : option residuals
%==========================================================================================
% This ver: 2021/10/16
% Authors: Yifan Li (yifan.li@manchester.ac.uk)
%          Ingmar Nolte (i.nolte@lancaster.ac.uk)
%          Manh Pham (m.c.pham@lancaster.ac.uk)
% Reference: Li, Y., Nolte, I., and Pham, M. C. (2021). Mixture-of-Lognormal 
%           Risk-Neutral Density Estimation Revisited: Asymptotics, Analytical Derivatives,
%           and the Mode Constraint
%========================================================================================== 


%Decomposing the parameter vector
if size(parm,1) < size(parm,2); parm = parm'; end % Use column representation
sigma_vec = parm(1:nmixture);
if nmixture > 1
    w_vec = [parm(1+nmixture:2*nmixture-1); 1 - sum(parm(1+nmixture:2*nmixture-1))];
    f_vec = [parm(2*nmixture:end); (ft - parm(2*nmixture:end)'*w_vec(1:end-1))/w_vec(end)];
else
    w_vec = 1; f_vec = ft;
end

%Computing the mixture-wise prices and option Greeks
GreeksCal1 = analyticGrad || analyticHess;
[fprices, delta, vega, Gamma, xi, eta] = BS1_f(f_vec','forward', K,rf,0,TTM,sigma_vec', o_type, GreeksCal1, analyticHess);

%Computing fitted price and MSE
N = length(oprice);
fprice=fprices*w_vec;
resid=oprice-fprice;
mse=resid'*resid/(2*N);

%Computing analytical gradient, hessian and per-obs gradient
if GreeksCal1
    Q_grad=[vega.*w_vec'  fprices(:,1:end-1)-fprices(:,end)-delta(:,end).*(f_vec(1:end-1)'-f_vec(end))  w_vec(1:end-1)'.*(delta(:,1:end-1)-delta(:,end))];
end

if analyticGrad
    grad=-Q_grad'*resid/N; % gradient
    gradObs = -Q_grad.*resid/N; % per-obs gradients
else
    grad = []; gradObs = []; 
end

if analyticHess
    hess = MLN_Hessian(parm,ft,nmixture,resid,Q_grad,delta,vega,Gamma,eta,xi)/N;
else
    hess = [];
end

end 

