function hess = MLN_Hessian(parm,F,nmixture,resid,Q_grad,delta,vega,Gamma,eta,xi)
%==========================================================================================
% This function computes the Hessian matrix of the minimization quadratic problem
% Q(theta)=(O(theta)-O_hat)'(O(theta)-O_hat)/2 w.r.t. theta, where O(theta) is the mixture
% of lognormal implied option prices, and O_hat are the observed option
% prices.
%
%
% INPUT:
%        parm        : a vector of parameters (sigma, weights, Futures) of the log-normal densities
%                        in a mixture
%                        parvec = c(sigma_1, sigma_2, ..., sigma_M,
%                                       w_1, w_2, ..., w_{M},
%                                       F_1, F_2, ..., F_{M}),
%                        where n is the number of lognormal densities in a mixture,
%                              w's and F's satisfy two constraints:
%                              1 = w_1 + ... + w_{M-1} + w_M,
%                              F = w_1*F_1 + ... + w_M*F_M
%                        The last condition is to ensure that the mixture of log-normal densities is risk-neutral
%                        Thus, length(parvec) = 3M-2
%        F             : current futures price used in the above constraint
%        nmixture      : number of component densities in a mixture.
%        fprices       : N-by-M vector of component-wise option prices
%        Q_grad        : N-by-(3M-2) Jacobian matrix: dO(theta)/dtheta
%        delta         : N-by-M matrix of component-wise option delta
%        vega          : N-by-M matrix of component-wise option vega
%        gamma         : N-by-M matrix of component-wise option gamma
%        eta           : N-by-M matrix of component-wise option eta, where eta is defined as d^2O(theta)/dsigma^2
%        xi            : N-by-M matrix of component-wise option xi, where
%                        xi is defined as d^2O(theta)/dsigma/dF
%
%
% OUTPUT:
%       hess           : (3M-2)x(3M-2) Hessian matrix defined as
%                         d^2Q(theta)/dtheta/dtheta'.
%==========================================================================================
% This ver: 2021/10/16
% Authors: Yifan Li (yifan.li@manchester.ac.uk)
%          Ingmar Nolte (i.nolte@lancaster.ac.uk)
%          Manh Pham (m.c.pham@lancaster.ac.uk)
% Reference: Li, Y., Nolte, I., and Pham, M. C. (2021). Mixture-of-Lognormal 
%           Risk-Neutral Density Estimation Revisited: Asymptotics, Analytical Derivatives,
%           and the Mode Constraint
%========================================================================================== 

%Use a row parameter vector
[u,v]=size(parm);
if u>v
    parm=parm';
end
K=length(parm);

%Isolate weight and component-wise F
w_vec=[parm(nmixture+1:2*nmixture-1)  1-sum(parm(nmixture+1:2*nmixture-1))];
f_vec=[parm(2*nmixture:3*nmixture-2) (F-w_vec(1:end-1)*parm(2*nmixture:3*nmixture-2)')/w_vec(end)];

%Calculate the "outerproduct of gradient part" of Hessian
hess=Q_grad'*Q_grad;
%Determine the range of the parameters in the parameter vector
sig_range=1:nmixture;
w_range=nmixture+1:2*nmixture-1;
f_range=2*nmixture:3*nmixture-2;
%Initializing the "observation-wise Hessian" part of Hessian
hmat=zeros(K,K);
%Calculate the second derivative w.r.t. sigma_i^2
hmat(sig_range,sig_range)=diag((w_vec.*xi)'*resid);
if nmixture>1
    %Calculate the second derivatives w.r.t. w_i  w_j
    hmat(w_range,w_range)= ivech((Gamma(:,end)/w_vec(end)*vech((f_vec(end)-f_vec(1:end-1))'*(f_vec(end)-f_vec(1:end-1)))')'*resid);
    %Calcualte the second derivative w.r.t. F_i F_j
    hmat(f_range,f_range)=ivech((Gamma(:,end)*vech(w_vec(1:end-1)'*w_vec(1:end-1)/w_vec(end))')'*resid)+diag((w_vec(1:end-1).*Gamma(:,1:end-1))'*resid);
    %Calculate the second derivative w.r.t. sigma_i w_j for i<M
    hmat(w_range,sig_range(1:end-1))=diag(vega(:,1:end-1)'*resid);
    %Calculate the second derivative w.r.t. sigma_i F_j for i<M
    hmat(f_range,sig_range(1:end-1))=diag((w_vec(1:end-1).*eta(:,1:end-1))'*resid);
    %Calculate the second derivative w.r.t. sigma_M w_i
    hmat(w_range,sig_range(end))=((f_vec(end)-f_vec(1:end-1)).*eta(:,end)-vega(:,end))'*resid;
    %Calculate the second derivative w.r.t. sigma_M F_i
    hmat(f_range,sig_range(end))=-(w_vec(1:end-1).*eta(:,end))'*resid;
    %Calculate the second derivative w.r.t. F_i w_j
    hmat(f_range,w_range)=reshape((Gamma(:,end)*reshape(-repmat(w_vec(1:end-1),nmixture-1,1)'/w_vec(end).*(f_vec(end)-repmat(f_vec(1:end-1),nmixture-1,1)),1,(nmixture-1)^2))'*resid,nmixture-1,nmixture-1) ...
        +diag((delta(:,1:end-1)-delta(:,end))'*resid);
end
%Replicate the lower triangular half to form a symmetric matrix
hmat=ivech(vech(hmat));
%Combine the two parts of Hessian
hess=-hmat+hess;
end


%========================================================================
%Utility functions from MFE toolbox of Kevin Sheppard
%========================================================================

function stackedData = vech(matrixData)
[k,~] = size(matrixData);

sel = tril(true(k));
stackedData = matrixData(sel);
end
function matrixData=ivech(stackedData)

if size(stackedData,2)>size(stackedData,1)
    stackedData=stackedData';
end

if size(stackedData,2)~=1
    error('STACKED_DATA must be a column vector.')
end

K2=size(stackedData,1);
K=(-1+sqrt(1+8*K2))/2;
if floor(K)~=K
    error(['The number of elemeents in STACKED_DATA must be conformable to' ...
        'the inverse vech operation.'])
end

matrixData=zeros(K);
pl=tril(true(K));
matrixData(pl)=stackedData;
diag_matrixData=diag(diag(matrixData));
matrixData=matrixData+matrixData'-diag_matrixData;
end
