function [oprice, Delta, Vega, Gamma, xi, eta] = ...
    BS1_f(u_price, u_price_type, strike, rf, divyield, TTM, sigma, o_type, GreeksCal1, GreeksCal2)
%==========================================================================================
% This function computes the Black-Scholes European option price
%
% INPUT:
%        u_price       : price of the underlying at time t
%        u_price_type  : Type of the underlying price, valid types are
%                        'spot'    -  spot price [default]
%                        'forward' -  forward/futures price
%        strike        : strike price of the option.
%        rf            : Annualized risk free rate, in decimals. Default = 0.
%        divyield      : Annualized dividend yield, in decimals. Default = 0.
%        TTM           : Time to maturity, as a fraction of a year (1 year = 360 days). Default = 1
%        sigma         : Annualized standard deviation of underlying asset's returns
%        o_type        : Option type, valid types are:
%                        'call'  - Call options
%                        'put'   - put options
%        GreeksCal1    : Logical, whether to compute Greeks or not. Default = true.
%                        If GreeksCal1 = false, Delta = Vega = Gamma = xi = eta = []
%        GreeksCal2    : Logical, whether to compute extended Greeks. Default = true.
%                        If GreekCal2 = false, Gamma = xi = eta = []
%
% OUTPUT:
%        oprice        : Price of options
%        Delta         : Option delta (dO/dS or dO/dF) 
%        Vega          : Option Vega  (dO/dsigma) (same for calls and puts)
%        Gamma         : Option Gamma (d^2 O/dS^2 or d^2 O/dF^2) (same for calls and puts)
%        xi            : Option xi    (d^2 O/dsigma^2 = d Vega/ dsigma)
%        eta           : Option eta   (d^2 O/dsigma/dS or d^2 O/dsigma/dF)
%
%
%==========================================================================================
% Black-Scholes formula for European Call option with strike K and TTM = tau
%
% % % If Spot price S of the underlying is provided
%       C = N(d1)*S*exp(-q*tau) - N(d2)*exp(-r*tau)*K,
% where
%       d1 = 1/(sigma*sqrt(tau))*[log(S/K) + (r-q+0.5*sigma^2)*tau]
%       d2 = d1 - sigma*sqrt(tau)
%       N(.)  : cumulative probability density of a standard normal distribution
%       r     : annualized (continuously compounded) risk free interest rate
%       q     : annualized (continuosly compounded) dividend yield
%
%
% % % If Forward price of the underlying is provided
%       C = N(d1)*exp(-r*tau)*F - N(d2)*exp(-r*tau)*K,
% where
%       d1 = 1/(sigma*sqrt(tau))*[log(F/K) + (0.5*sigma^2)*tau]
%       d2 = d1 - sigma*sqrt(tau)
%       N(.) and r are defined as above.
%
% Here, F = exp((r-q)*tau)*S
%==========================================================================================
% Black-Scholes formula for European Put option with strike K and TTM = tau
%
% % % If Spot price S of the underlying is provided
% Put-call parity: P + S*exp(-q*tau) = C + exp(-r*tau)*K
% thus,
%       P = exp(-r*tau)*K*(1- N(d2)) - S*exp(-q*tau)*(1 - N(d1))
%         = N(-d2)*exp(-r*tau)*K - N(-d1)*S*exp(-q*tau)
%
% % % If Forward price S of the underlying is provided
% Put-call parity: P + F*exp(-r*tau) = C + exp(-r*tau)*K
% thus,
%       P = exp(-r*tau)*K*(1- N(d2)) - F*exp(-r*tau)*(1 - N(d1))
%         = N(-d2)*exp(-r*tau)*K - N(-d1)*F*exp(-r*tau)
%==========================================================================================
% This ver: 2021/10/16
% Authors: Yifan Li (yifan.li@manchester.ac.uk)
%          Ingmar Nolte (i.nolte@lancaster.ac.uk)
%          Manh Pham (m.c.pham@lancaster.ac.uk)
% Reference: Li, Y., Nolte, I., and Pham, M. C. (2021). Mixture-of-Lognormal 
%           Risk-Neutral Density Estimation Revisited: Asymptotics, Analytical Derivatives,
%           and the Mode Constraint
%========================================================================================== 


%===================
% Input checking
%===================
% u_price
u_price(u_price < 0) = NaN;

% u_price_type
if nargin < 2 || isempty(u_price_type)
    u_price_type = 'spot';
end

if ~ismember(u_price_type, {'spot', 'forward'})
    error('u_price_type must be a string and one of: ''spot'' or ''forward''.');
end

% rf
if nargin < 4 || isempty(rf)
    rf = 0;
end

% divyield
if nargin < 5 || isempty(divyield)
    divyield = 0;
end

% TTM
if nargin < 6 || isempty(TTM)
    TTM = 1;
end

% o_type
if nargin < 8 || isempty(o_type)
    o_type = 'call';
end

if ~ismember(o_type, {'call', 'put'})
    error('o_type must be a string and one of: ''call'' or ''put''.');
end

% GreeksCal
if ~GreeksCal1
    Delta = []; Vega = []; Gamma = []; xi = []; eta = [];
end

if ~GreeksCal2
    Gamma = []; xi = []; eta = [];
end
%===================
% Calculations
%===================

if strcmp(u_price_type, 'spot')
    % d1
    d1 = (log(u_price./strike) + (rf - divyield + 0.5*sigma.^2).*TTM)./(sigma.*sqrt(TTM));
    
    % d2
    d2 = d1 - sigma.*sqrt(TTM);
    
    % Option price
    Nd1 = normcdf(d1);
    Nd2 = normcdf(d2);
    nd1 = normpdf(d1);
    discount = exp(-rf.*TTM);
    discount2 = exp(-divyield.*TTM);
    if strcmp(o_type, 'call')
        oprice = Nd1.*u_price.*discount2 - Nd2.*discount.*strike;
    else
        oprice  = (1-Nd2).*discount.*strike - (1-Nd1).*u_price.*discount2;
    end
    
    % Option Greeks
    if GreeksCal1 % Only compute Greeks if GreeksCal is true
        if strcmp(o_type, 'call')
            Delta = Nd1;
        else
            Delta = Nd1 - 1;
        end
        Vega = u_price.*nd1.*sqrt(TTM);
        
        if GreeksCal2 %Extended Greeks
            Gamma = nd1./u_price./sqrt(TTM)./sigma;
            xi=Vega.*d1.*d2./sigma;
            eta=-nd1.*d2./sigma;
        end
    end
    
else
    
    % d1
    d1 = (log(u_price./strike) + (0.5*sigma.^2).*TTM)./(sigma.*sqrt(TTM));
    
    % d2
    d2 = d1 - sigma.*sqrt(TTM);
    
    % Option price
    Nd1 = normcdf(d1);
    Nd2 = normcdf(d2);
    nd1 = normpdf(d1);
    discount = exp(-rf.*TTM);
    if strcmp(o_type, 'call')
        oprice = discount.*(Nd1.*u_price - Nd2.*strike);
    else
        oprice  = discount.*((1-Nd2).*strike - (1-Nd1).*u_price);
    end
    
    % Option Greeks
    if GreeksCal1
        if strcmp(o_type, 'call')
            Delta = discount.*Nd1;
        else
            Delta = discount.*(Nd1-1);
        end
        Vega = discount.*u_price.*nd1.*sqrt(TTM);
        
        if GreeksCal2
            Gamma = discount.*nd1./u_price./sqrt(TTM)./sigma;
            xi=Vega.*d1.*d2./sigma;
            eta=-discount.*nd1.*d2./sigma;
        end
    end
    
end % end if strcmp(u_price_type, 'spot')

end % end function
